<?php
/**
 * Plugin Name: StablePay for WooCommerce
 * Plugin URI: https://stablepay.co
 * Description: 为 WooCommerce 商城接入 StablePay 稳定币支付能力，支持 USDT、USDC 等多种稳定币支付
 * Version: 1.0.0
 * Author: StablePay
 * Author URI: https://stablepay.co
 * License: MIT
 * Text Domain: stablepay-woocommerce
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * WC requires at least: 6.0
 * WC tested up to: 10.3
 */

// 防止直接访问
if (!defined('ABSPATH')) {
    exit;
}

// 定义插件常量
define('STABLEPAY_WC_VERSION', '1.0.0');
define('STABLEPAY_WC_PLUGIN_FILE', __FILE__);
define('STABLEPAY_WC_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('STABLEPAY_WC_PLUGIN_URL', plugin_dir_url(__FILE__));
define('STABLEPAY_WC_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * 声明 WooCommerce 兼容性
 * - HPOS (High-Performance Order Storage) / Custom Order Tables
 * - Cart and Checkout Blocks
 */
add_action('before_woocommerce_init', function() {
    // 声明 HPOS 兼容性 (WooCommerce 8.2+)
    if (class_exists('\Automattic\WooCommerce\Utilities\FeaturesUtil')) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
    }
    // 声明 Cart/Checkout Blocks 兼容性 (WooCommerce 8.3+)
    if (class_exists('\Automattic\WooCommerce\Utilities\FeaturesUtil')) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('cart_checkout_blocks', __FILE__, true);
    }
});

/**
 * 检查依赖
 */
function stablepay_wc_check_dependencies() {
    // 检查 PHP 版本
    if (version_compare(PHP_VERSION, '7.4', '<')) {
        add_action('admin_notices', function() {
            echo '<div class="error"><p>';
            echo sprintf(
                __('StablePay for WooCommerce 需要 PHP 7.4 或更高版本，当前版本为 %s', 'stablepay-woocommerce'),
                PHP_VERSION
            );
            echo '</p></div>';
        });
        return false;
    }

    // 检查 WordPress 版本
    global $wp_version;
    if (version_compare($wp_version, '5.8', '<')) {
        add_action('admin_notices', function() use ($wp_version) {
            echo '<div class="error"><p>';
            echo sprintf(
                __('StablePay for WooCommerce 需要 WordPress 5.8 或更高版本，当前版本为 %s', 'stablepay-woocommerce'),
                $wp_version
            );
            echo '</p></div>';
        });
        return false;
    }

    // 检查 WooCommerce 是否已安装并激活
    if (!class_exists('WooCommerce')) {
        add_action('admin_notices', function() {
            echo '<div class="error"><p>';
            echo __('StablePay for WooCommerce 需要安装并激活 WooCommerce 插件', 'stablepay-woocommerce');
            echo '</p></div>';
        });
        return false;
    }

    // 检查 WooCommerce 版本
    if (defined('WC_VERSION') && version_compare(WC_VERSION, '6.0', '<')) {
        add_action('admin_notices', function() {
            echo '<div class="error"><p>';
            echo sprintf(
                __('StablePay for WooCommerce 需要 WooCommerce 6.0 或更高版本，当前版本为 %s', 'stablepay-woocommerce'),
                WC_VERSION
            );
            echo '</p></div>';
        });
        return false;
    }

    // 检查 cURL 扩展
    if (!function_exists('curl_init')) {
        add_action('admin_notices', function() {
            echo '<div class="error"><p>';
            echo __('StablePay for WooCommerce 需要启用 PHP cURL 扩展', 'stablepay-woocommerce');
            echo '</p></div>';
        });
        return false;
    }

    return true;
}

/**
 * 初始化插件
 */
function stablepay_wc_init() {
    // 检查依赖
    if (!stablepay_wc_check_dependencies()) {
        return;
    }

    // 加载文本域
    load_plugin_textdomain('stablepay-woocommerce', false, dirname(STABLEPAY_WC_PLUGIN_BASENAME) . '/languages');

    // 加载核心类文件
    require_once STABLEPAY_WC_PLUGIN_DIR . 'includes/class-stablepay-logger.php';
    require_once STABLEPAY_WC_PLUGIN_DIR . 'includes/class-stablepay-api-client.php';
    require_once STABLEPAY_WC_PLUGIN_DIR . 'includes/class-stablepay-webhook.php';
    require_once STABLEPAY_WC_PLUGIN_DIR . 'includes/class-stablepay-refund.php';
    require_once STABLEPAY_WC_PLUGIN_DIR . 'includes/class-stablepay-gateway.php';

    // 注册支付网关
    add_filter('woocommerce_payment_gateways', 'stablepay_wc_add_gateway');

    // 修复零小数货币的价格显示（如 JPY, IDR, VND, KRW 应显示为整数而非 XXX.00）
    add_filter('wc_price_args', 'stablepay_wc_fix_zero_decimal_currency_display');

    // 加载前端样式
    add_action('wp_enqueue_scripts', 'stablepay_wc_enqueue_styles');

    // 注册 Blocks 支付支持 (WooCommerce 8.0+)
    add_action('woocommerce_blocks_loaded', 'stablepay_wc_blocks_support');

    // 初始化 Webhook 处理器（独立于网关实例化，确保 REST API 路由正确注册）
    stablepay_wc_init_webhook_handler();
}
add_action('plugins_loaded', 'stablepay_wc_init');

/**
 * 初始化 Webhook 处理器
 *
 * 必须在 plugins_loaded 时初始化，确保 rest_api_init hook 能被正确注册
 * 不能依赖网关实例化，因为网关可能在 rest_api_init 之后才被实例化
 */
function stablepay_wc_init_webhook_handler() {
    // 从 WooCommerce 设置中获取 webhook_secret
    $gateway_settings = get_option('woocommerce_stablepay_settings', array());
    $webhook_secret = isset($gateway_settings['webhook_secret']) ? $gateway_settings['webhook_secret'] : '';

    // 获取日志配置
    $log_enabled = isset($gateway_settings['debug']) && $gateway_settings['debug'] === 'yes';
    $log_level = isset($gateway_settings['log_level']) ? $gateway_settings['log_level'] : 'info';

    // 创建日志记录器
    $logger = new StablePay_Logger($log_enabled, $log_level);

    // 初始化 Webhook 处理器（即使 secret 为空也初始化，这样路由至少会被注册）
    global $stablepay_webhook_handler;
    $stablepay_webhook_handler = new StablePay_Webhook_Handler($webhook_secret, $logger);

    // 注意：不在这里记录日志，因为 plugins_loaded 在每次请求时都会触发
    // 会产生大量重复日志，淹没正常业务日志
}

/**
 * 注册 WooCommerce Blocks 支付支持
 */
function stablepay_wc_blocks_support() {
    if (!class_exists('Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType')) {
        return;
    }

    require_once STABLEPAY_WC_PLUGIN_DIR . 'includes/class-stablepay-blocks-support.php';

    add_action(
        'woocommerce_blocks_payment_method_type_registration',
        function($payment_method_registry) {
            $payment_method_registry->register(new WC_StablePay_Blocks_Support());
        }
    );
}

/**
 * 加载前端样式
 */
function stablepay_wc_enqueue_styles() {
    // 仅在结账页面加载
    if (is_checkout() || is_cart()) {
        wp_enqueue_style(
            'stablepay-woocommerce-frontend',
            STABLEPAY_WC_PLUGIN_URL . 'assets/css/frontend.css',
            array(),
            STABLEPAY_WC_VERSION
        );
    }
}

/**
 * 添加 StablePay 支付网关到 WooCommerce
 *
 * @param array $gateways 现有支付网关列表
 * @return array 包含 StablePay 的支付网关列表
 */
function stablepay_wc_add_gateway($gateways) {
    $gateways[] = 'WC_StablePay_Gateway';
    return $gateways;
}

/**
 * 插件激活钩子
 */
function stablepay_wc_activate() {
    // 检查依赖
    if (!stablepay_wc_check_dependencies()) {
        wp_die(__('无法激活插件：依赖检查失败', 'stablepay-woocommerce'));
    }

    // 创建日志目录（如果需要）
    $upload_dir = wp_upload_dir();
    $log_dir = $upload_dir['basedir'] . '/stablepay-logs';
    if (!file_exists($log_dir)) {
        wp_mkdir_p($log_dir);
    }

    // 刷新重写规则
    flush_rewrite_rules();
}
register_activation_hook(__FILE__, 'stablepay_wc_activate');

/**
 * 插件停用钩子
 */
function stablepay_wc_deactivate() {
    // 清理临时数据
    delete_transient('stablepay_webhook_events');

    // 刷新重写规则
    flush_rewrite_rules();
}
register_deactivation_hook(__FILE__, 'stablepay_wc_deactivate');

/**
 * 插件链接
 */
function stablepay_wc_plugin_links($links) {
    $plugin_links = array(
        '<a href="' . admin_url('admin.php?page=wc-settings&tab=checkout&section=stablepay') . '">' . __('设置', 'stablepay-woocommerce') . '</a>',
        '<a href="https://docs.stablepay.co" target="_blank">' . __('文档', 'stablepay-woocommerce') . '</a>',
        '<a href="https://stablepay.co/support" target="_blank">' . __('支持', 'stablepay-woocommerce') . '</a>',
    );
    return array_merge($plugin_links, $links);
}
add_filter('plugin_action_links_' . STABLEPAY_WC_PLUGIN_BASENAME, 'stablepay_wc_plugin_links');

/**
 * 修复零小数货币的价格显示
 *
 * WooCommerce 默认使用全局设置的小数位数显示所有货币价格
 * 但 JPY、IDR、VND、KRW 等货币应该显示为整数（无小数）
 *
 * @param array $args 价格格式化参数
 * @return array 修改后的参数
 */
function stablepay_wc_fix_zero_decimal_currency_display($args) {
    // 零小数位货币列表
    $zero_decimal_currencies = array('JPY', 'IDR', 'VND', 'KRW');

    // 获取当前货币
    $currency = !empty($args['currency']) ? strtoupper($args['currency']) : strtoupper(get_woocommerce_currency());

    // 如果是零小数货币，设置小数位数为 0
    if (in_array($currency, $zero_decimal_currencies)) {
        $args['decimals'] = 0;
    }

    return $args;
}
